/*
 * IBM Confidential OCO Source Material
 * 5639-D57 (C) COPYRIGHT International Business Machines Corp. 2002.
 * The source code for this program is not published or otherwise divested
 * of its trade secrets, irrespective of what has been deposited with the
 * U.S. Copyright Office.
 */

import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.util.Properties;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NamedNodeMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

//import org.apache.xml.serialize.OutputFormat;
//import org.apache.xml.serialize.Serializer;
//import org.apache.xml.serialize.XMLSerializer;

/**
 * This class converts a configuration document in XML to an appropriate format
 * for schema validation. It performs two following tasks.
 * 1) In the root element, <xmi:XMI>, add new attribute, xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
 * 2) Replace all xmi:type attribute name to xsi:type.
 */
public class XMItoXSIType {
    private DocumentBuilder parser;
    private String xmlFileName;
    private String outputFileName;

    private String ATTR_XMLNS_XSI = "xmlns:xsi";
    private String ATTRVAL_XMLNS_XSI = "http://www.w3.org/2001/XMLSchema-instance";

    private String ATTR_XMI_TYPE = "xmi:type";
    private String ATTR_XSI_TYPE = "xsi:type";

    private Document doc;

    /**
     * Constructor.
     */
    public XMItoXSIType(String xmlFileName) {
        this.xmlFileName = xmlFileName;

        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            parser = factory.newDocumentBuilder();
		}
		catch (Exception e) {
			System.err.println("error: Unable to instantiate a parser.");
		}
    } // Constructor

    /**
     * Perform the conversion operation.
     *
     * @return A converted DOM document. Null if conversion's failed.
     */
    public Document convert() {
        if (parser == null) {
            return null;
        }
    	try {
    	    doc = parser.parse(xmlFileName);
    	}
    	catch (Exception e) {
    	    System.err.println("XMItoXSIType: Failed to parse \"" + xmlFileName + "\"");
    	    return null;
    	}
    
    	convertIt(doc);
    	return doc;
    } // convert

    /**
     * Recursive function.
     */
    private void convertIt(Node node) {
    	if (node == null) {
            return;
        }

        short type = node.getNodeType();
	    if (type == Node.ELEMENT_NODE) {
	        Element element = (Element)node;
    	    NamedNodeMap attrs = node.getAttributes();
    	    int noOfAttr = attrs.getLength();
    	    if (element.getTagName().equals("xmi:XMI")) {
        		boolean foundIt = false;
        		for (int i=0; i<noOfAttr; i++) {
        		    Attr attr = (Attr)attrs.item(i);
        		    if (attr.getName().equals(ATTR_XMLNS_XSI)) {
            			foundIt = true;
            			break;
        		    }
        		} // for

		        if (foundIt == false) {
        		    element.setAttribute(ATTR_XMLNS_XSI, ATTRVAL_XMLNS_XSI);
        		}
    	    }
	        else {
        		// Replace all xmi:type to xsi:type.
        		for (int i=0; i<noOfAttr; i++) {
        		    Attr attr = (Attr)attrs.item(i);
        		    if (attr.getName().equals(ATTR_XMI_TYPE)) {
        			String attrValue = attr.getValue();
        			element.removeAttribute(ATTR_XMI_TYPE);
        			element.setAttribute(ATTR_XSI_TYPE, attrValue);
        		    }
        		} // for
    	    }
	    }

    	Node child = node.getFirstChild();
    	while (child != null) {
    	    convertIt(child);
    	    child = child.getNextSibling();
    	} // while
    } // convertIt

    /**
     * Get the converted XML file name.
     */
    public String getOutputFileName() {
    	if (outputFileName == null) {
    	    return null;
    	}
    
    	return new String(outputFileName);
    } // getOutputFileName

    /**
     * Specify the output file name for converted XML file.
     */
    public void setOutputFileName(String outputFileName) {
    	if (outputFileName != null) {
    	    this.outputFileName = new String(outputFileName);
    	}
    } // setOutputFileName

    /**
     * Serialize DOM to a file.
     */
    public void save() throws FileNotFoundException, IOException, TransformerException, TransformerConfigurationException {
    	if (doc == null) {
    	    return;
    	}
    
    	if (outputFileName == null) {
    	    outputFileName = xmlFileName + ".tmp";
    	}
    
    	PrintWriter writer = new PrintWriter(new FileOutputStream(outputFileName));
//    	OutputFormat format  = new OutputFormat(doc);
//        XMLSerializer serial = new XMLSerializer(writer, format);
//    	serial.asDOMSerializer();		// As a DOM Serializer
//    	serial.serialize(doc.getDocumentElement());

        TransformerFactory factory = TransformerFactory.newInstance();
        Transformer serializer = factory.newTransformer();
        Properties props = new Properties();
        props.put(OutputKeys.METHOD, "xml");
        props.put(OutputKeys.OMIT_XML_DECLARATION, "no");
        props.put(OutputKeys.VERSION, "1.0");
        props.put(OutputKeys.ENCODING, "UTF-8");
        props.put(OutputKeys.INDENT, "true");
        serializer.setOutputProperties(props);
        serializer.transform(new DOMSource(doc), new StreamResult(writer));

    } // save

    /**
     * Standalone test program.
     */
    public static void main(String[] args) throws Exception {
    	if (args.length != 1) {
    	    System.err.println("Usage: java XMItoXSIType <xml file>");
    	    System.exit(-1);
    	}
    
    	XMItoXSIType converter = new XMItoXSIType(args[0]);
    	converter.convert();
    	converter.save();
    }

} // XMItoXSIType
